#
# Copyright (c) 2006 Mellanox Technologies. All rights reserved.
#
# This Software is licensed under one of the following licenses:
#
# 1) under the terms of the "Common Public License 1.0" a copy of which is
#    available from the Open Source Initiative, see
#    http://www.opensource.org/licenses/cpl.php.
#
# 2) under the terms of the "The BSD License" a copy of which is
#    available from the Open Source Initiative, see
#    http://www.opensource.org/licenses/bsd-license.php.
#
# 3) under the terms of the "GNU General Public License (GPL) Version 2" a
#    copy of which is available from the Open Source Initiative, see
#    http://www.opensource.org/licenses/gpl-license.php.
#
# Licensee has the right to choose one of the above licenses.
#
# Redistributions of source code must retain the above copyright
# notice and one of the license notices.
#
# Redistributions in binary form must reproduce both the above copyright
# notice, one of the license notices in the documentation
# and/or other materials provided with the distribution.
#
#  $Id: build_env.sh 9750 2006-10-09 17:09:31Z vlad $
#
# Description: OFED building environment

PACKAGE="OFED"

ARCH=${ARCH:-$(uname -m)}
K_VER=${K_VER:-$(uname -r)}
K_SRC=${K_SRC:-/lib/modules/${K_VER}/build}
KERNEL_TREE_ROOT=${K_SRC%*/*}

RPMK_VER=$(echo -n ${K_VER} | tr -s '-' '_')
IB_PORTS_NUM=2

NULL=/dev/null
curr_dir=$PWD

RM=/bin/rm
CP=/bin/cp
MV=/bin/mv
MKDIR=/bin/mkdir
RPM=/bin/rpm
IFCONFIG=/sbin/ifconfig
CHMOD=/bin/chmod
LSMOD=/sbin/lsmod
WHICH=/usr/bin/which
FIND=/usr/bin/find
DF=/bin/df
REBOOT="/sbin/shutdown -r 0"
SUPPORTED=0
NEED_REBOOT=0
EXECUTING=0
RUNNING_CLEANUP=0
INCLUDE_IFCFG=${INCLUDE_IFCFG:-1}

BASIC=1
HPC=2
ALL=3
CUSTOM=4

prog=`basename $0`

# Disk space required for building ${PACKAGE} RPMs (300 MB)
BUILD_SPACE_REQ=307200

# Disk space required for installing ${PACKAGE} RPMs (200 MB)
RUN_SPACE_REQ=204800

# Default vendor is "ts" (second - mtl)
VENDOR="ts"
build_arch=$(rpm --eval '%{_target_cpu}')
mandir=$(rpm --eval '%{_mandir}')
sysconfdir=$(rpm --eval '%{_sysconfdir}')

RPM_DIR_DFT="/var/tmp/${PACKAGE}RPM"
RPM_DIR=${RPM_DIR:-"${RPM_DIR_DFT}"}
if [ -d $RPM_DIR ]; then
    find $RPM_DIR -type d | sort -r | xargs rmdir > $NULL 2>&1
fi
mkdir -p $RPM_DIR/{BUILD,RPMS,SOURCES,SPECS,SRPMS}

# Default Fortran compiler for MPI installation
MPI_SUPPORTED_COMPILERS="gcc pgi intel pathscale"

# OFED kernel packages
KERNEL_TECHNOLOGY_PREVIEW=""

# Basic kernel modules
BASIC_IB_KERNEL_PACKAGES="ib_verbs ib_mthca ib_ipoib"

# Ipath
case ${ARCH} in
        x86_64)
                case ${K_VER} in
                        2.6.5*|2.6.9-22*|2.6.9-34*|2.6.9-42*|2.6.9-55*|2.6.16.*-*-*|2.6.1[7-9]*|2.6.20*)
                        BASIC_IB_KERNEL_PACKAGES="${BASIC_IB_KERNEL_PACKAGES} ib_ipath"
                        ;;
                esac
        ;;
        ppc64)
                case ${K_VER} in
                        2.6.16.*-*-*|2.6.1[7-9]*|2.6.20*)
                        BASIC_IB_KERNEL_PACKAGES="${BASIC_IB_KERNEL_PACKAGES} ib_ipath"
                        ;;
                esac
        ;;
esac

# Ehca
case ${ARCH} in
        ppc|ppc64|powerpc)
                case ${K_VER} in
                    2.6.1[6-9]* | 2.6.20* | 2.6.9-55*)
                    BASIC_IB_KERNEL_PACKAGES="${BASIC_IB_KERNEL_PACKAGES} ib_ehca"
                    ;;
                esac
        ;;
esac

# Other kernel modules
IB_KERNEL_PACKAGES=""

# SDP
IB_KERNEL_PACKAGES="${IB_KERNEL_PACKAGES} ib_sdp"

# SRP
IB_KERNEL_PACKAGES="${IB_KERNEL_PACKAGES} ib_srp"

# Iser
# Currently iSER is supported only on SLES10 & RHEL5
case ${K_VER} in
        2.6.16.*-*-*|2.6.*.el5|2.6.9-[3-5]*.EL*)
        IB_KERNEL_PACKAGES="${IB_KERNEL_PACKAGES} ib_iser"
        ;;
esac

# Vnic
case ${K_VER} in
        2.6.9-34*|2.6.9-42*|2.6.9-55*|2.6.16.*-*-*|2.6.19*)
        IB_KERNEL_PACKAGES="${IB_KERNEL_PACKAGES} vnic"
        ;;
esac

# RDS
case ${K_VER} in
        2.6.5*)
        ;;
        *)
        IB_KERNEL_PACKAGES="${IB_KERNEL_PACKAGES} rds"
        ;;
esac

# Cxgb3
case ${K_VER} in
        2.6.13-1[5-9]*)
        ;;
        *)
        IB_KERNEL_PACKAGES="${IB_KERNEL_PACKAGES} cxgb3"
        ;;
esac

IB_KERNEL_PACKAGES="$BASIC_IB_KERNEL_PACKAGES $IB_KERNEL_PACKAGES $KERNEL_TECHNOLOGY_PREVIEW"

IB_KERNEL_UTILS=""

IB_BASIC_PACKAGES="kernel-ib kernel-ib-devel libibverbs libibverbs-devel libibverbs-utils"
IB_BASIC_PACKAGES="$IB_BASIC_PACKAGES libibcm libibcm-devel libmthca libmthca-devel perftest mstflint"

# Add libipathverbs package only if IPATH driver can be installed
if (echo -n ${IB_KERNEL_PACKAGES} | grep -w "ib_ipath" > $NULL); then
    IB_BASIC_PACKAGES="$IB_BASIC_PACKAGES libipathverbs libipathverbs-devel"
fi

# Add libehca package only if EHCA driver can be installed
if (echo -n ${IB_KERNEL_PACKAGES} | grep -w "ib_ehca" > $NULL); then
    IB_BASIC_PACKAGES="$IB_BASIC_PACKAGES libehca libehca-devel"
fi

# Add libcxgb3 package only if CXGB3 driver can be installed
if (echo -n ${IB_KERNEL_PACKAGES} | grep -w "cxgb3" > $NULL); then
    IB_BASIC_PACKAGES="$IB_BASIC_PACKAGES libcxgb3 libcxgb3-devel"
fi

# Add qvnictools package only if VNIC driver can be installed
if (echo -n ${IB_KERNEL_PACKAGES} | grep -w "vnic" > $NULL); then
    IB_BASIC_PACKAGES="$IB_BASIC_PACKAGES qlvnictools"
fi

MISC_PACKAGES="ofed-docs ofed-scripts"

# Initialize IB_ALL_PACKAGES
IB_ALL_PACKAGES="$IB_BASIC_PACKAGES $MISC_PACKAGES"

# Add libsdp and sdpnetstat packages only if SDP module can be installed
if (echo -n ${IB_KERNEL_PACKAGES} | grep -w "ib_sdp" > $NULL); then
    IB_ALL_PACKAGES="$IB_ALL_PACKAGES libsdp sdpnetstat"
fi

# Add rds-tools package only if RDS can be installed
if (echo -n ${IB_KERNEL_PACKAGES} | grep -w "rds" > $NULL); then
    IB_ALL_PACKAGES="$IB_ALL_PACKAGES rds-tools"
fi

IB_ALL_PACKAGES="$IB_ALL_PACKAGES srptools ipoibtools tvflash"
IB_ALL_PACKAGES="$IB_ALL_PACKAGES libibcommon libibcommon-devel libibmad libibmad-devel libibumad libibumad-devel"
IB_ALL_PACKAGES="$IB_ALL_PACKAGES libopensm libopensm-devel opensm"
IB_ALL_PACKAGES="$IB_ALL_PACKAGES libosmcomp libosmcomp-devel libosmvendor libosmvendor-devel"
IB_ALL_PACKAGES="$IB_ALL_PACKAGES openib-diags"
IB_ALL_PACKAGES="$IB_ALL_PACKAGES librdmacm librdmacm-devel librdmacm-utils"


# Dapl is not supported on the PPC64 arcitecture
case $(uname -m) in
    ppc64)
    ;;
    *)
        IB_ALL_PACKAGES="$IB_ALL_PACKAGES dapl dapl-devel dapl-utils"
    ;;
esac

# Add bonding support for RH4.0U3 and SLES10
case ${K_VER} in
    2.6.16.*-*-*|2.6.9-34.EL*|2.6.9-42.EL*|2.6.9-55*|2.6.18-*.*)
        IB_ALL_PACKAGES="$IB_ALL_PACKAGES ib-bonding"
    ;;
esac

MANAGEMENT_LIBS="libibcommon libibmad libibumad"
OPENSM_PACKAGES="$MANAGEMENT_LIBS libopensm libopensm-devel opensm"
DIAGS_PACKAGES="$MANAGEMENT_LIBS openib-diags ibutils"

###
### Installation Types
###

BASIC_PACKAGES="${IB_BASIC_PACKAGES} ${MISC_PACKAGES}"
HPC_PACKAGES="${BASIC_PACKAGES} ${MANAGEMENT_LIBS} openib-diags mpi-selector mvapich mvapich2 openmpi mpitests ibutils"
HPC_PACKAGES="${HPC_PACKAGES} librdmacm librdmacm-devel librdmacm-utils dapl dapl-devel dapl-utils"
# The list of ALL_PACKAGES should preserve the order of dependencies
ALL_PACKAGES="${IB_ALL_PACKAGES} mpi-selector mvapich mvapich2 openmpi mpitests ibutils"

#Packages that are not displayed in custom menu
HIDDEN_PACKAGES="open-iscsi"

PREV_RELEASE_PACKAGES="opensm-devel opensm-libs ibtsal openib mpi_ncsa mpi_osu thca ib-osm osm ibadm ib-diags"
PREV_RELEASE_PACKAGES="$PREV_RELEASE_PACKAGES diags ibgdiag ibdiag ib-management"
PREV_RELEASE_PACKAGES="$PREV_RELEASE_PACKAGES ib-verbs ib-ipoib ib-cm ib-sdp"
PREV_RELEASE_PACKAGES="$PREV_RELEASE_PACKAGES ib-dapl udapl udapl-devel"
PREV_RELEASE_PACKAGES="$PREV_RELEASE_PACKAGES libdat libibat ib-kdapl ib-srp ib-srp_target"
PREV_RELEASE_PACKAGES="$PREV_RELEASE_PACKAGES oiscsi-iser-support libipathverbs libipathverbs-devel"
PREV_RELEASE_PACKAGES="$PREV_RELEASE_PACKAGES libehca libehca-devel"

RH_PACKAGES="udapl udapl-devel dapl dapl-devel libibcm libibcm-devel libibcommon libibcommon-devel"
RH_PACKAGES="$RH_PACKAGES libibmad libibmad-devel libibumad libibumad-devel"
RH_PACKAGES="$RH_PACKAGES libibverbs libibverbs-devel libibverbs-utils"
RH_PACKAGES="$RH_PACKAGES libipathverbs libipathverbs-devel libmthca libmthca-devel libehca libehca-devel"
RH_PACKAGES="$RH_PACKAGES libsdp librdmacm librdmacm-devel librdmacm-utils openib"
RH_PACKAGES="$RH_PACKAGES openib-diags openib-mstflint openib-perftest openib-srptools openib-tvflash"
RH_PACKAGES="$RH_PACKAGES openmpi openmpi-devel openmpi-libs opensm opensm-devel opensm-libs"
RH_PACKAGES="$RH_PACKAGES ibutils ibutils-devel ibutils-libs"

# Source path definitions
PKG_SOURCE_DIR=${PWD}

PROPEL=${PKG_SOURCE_DIR}/SOURCES/propel.sh

# Set Distribuition dependency environment
dist_rpm=""
dist_rpm=$($RPM -qf /etc/issue 2> $NULL)

if [ -f /etc/SuSE-release ]; then
    DISTRIBUTION="SuSE"
elif [ -f /etc/fedora-release ]; then
        case ${K_VER} in
                *fc6*)
                DISTRIBUTION="fedora6"
                ;;
                *)
                DISTRIBUTION="fedora"
                ;;
        esac
elif [ -f /etc/rocks-release ]; then
    DISTRIBUTION="Rocks"
elif [ -f /etc/redhat-release ]; then
        case ${K_VER} in
                *el5*)
                DISTRIBUTION="redhat5"
                ;;
                *)
                case $(cat /etc/redhat-release) in
                        *"release 5"*)
                        DISTRIBUTION="redhat5"
                        ;;
                        *)
                        DISTRIBUTION="redhat"
                        ;;
                esac
                ;;
        esac
elif [ -f /etc/debian_version ]; then
    DISTRIBUTION="debian"
else
    DISTRIBUTION=$(ls /etc/*-release | head -n 1 | xargs -iXXX basename XXX -release 2> $NULL)
    [ -z "${DISTRIBUTION}" ] && DISTRIBUTION="unsupported"
fi

if [ -z "dist_rpm" ]; then
    dist_rpm="unsupported"
fi

RPMS=${PKG_SOURCE_DIR}/RPMS/${dist_rpm}
SRPMS=${PKG_SOURCE_DIR}/SRPMS

if [ ! -d ${RPMS} ]; then
    $MKDIR -p ${RPMS} || exit 1
fi

OFA_USER_SRC_RPM=$(/bin/ls -1 ${SRPMS}/ofa_user*.src.rpm 2> $NULL)
if [[ -n "${OFA_USER_SRC_RPM}" && -s ${OFA_USER_SRC_RPM} ]]; then
    OFA_USER_NAME=$(rpm --queryformat "[%{NAME}]" -qp $OFA_USER_SRC_RPM)
    OFA_USER_VER=$(rpm --queryformat "[%{VERSION}]" -qp $OFA_USER_SRC_RPM)
    OFA_USER_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $OFA_USER_SRC_RPM)
    OFA_USER_VERSION=${OFA_USER_VER}-${OFA_USER_REL}
fi

OFA_KERNEL_SRC_RPM=$(/bin/ls -1 ${SRPMS}/ofa_kernel*.src.rpm 2> $NULL)
if [[ -n "${OFA_KERNEL_SRC_RPM}" && -s ${OFA_KERNEL_SRC_RPM} ]]; then
    OFA_KERNEL_NAME=$(rpm --queryformat "[%{NAME}]" -qp $OFA_KERNEL_SRC_RPM)
    OFA_KERNEL_VER=$(rpm --queryformat "[%{VERSION}]" -qp $OFA_KERNEL_SRC_RPM)
    OFA_KERNEL_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $OFA_KERNEL_SRC_RPM)
    OFA_KERNEL_VERSION=${OFA_KERNEL_VER}-${OFA_KERNEL_REL}
fi

DOCS_SRC_RPM=$(/bin/ls -1 ${SRPMS}/ofed-docs*.src.rpm 2> $NULL)
if [[ -n "${DOCS_SRC_RPM}" && -s ${DOCS_SRC_RPM} ]]; then
    DOCS_NAME=$(rpm --queryformat "[%{NAME}]" -qp $DOCS_SRC_RPM)
    DOCS_VER=$(rpm --queryformat "[%{VERSION}]" -qp $DOCS_SRC_RPM)
    DOCS_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $DOCS_SRC_RPM)
    DOCS_VERSION=${DOCS_VER}-${DOCS_REL}
fi

SCRIPTS_SRC_RPM=$(/bin/ls -1 ${SRPMS}/ofed-scripts*.src.rpm 2> $NULL)
if [[ -n "${SCRIPTS_SRC_RPM}" && -s ${SCRIPTS_SRC_RPM} ]]; then
    SCRIPTS_NAME=$(rpm --queryformat "[%{NAME}]" -qp $SCRIPTS_SRC_RPM)
    SCRIPTS_VER=$(rpm --queryformat "[%{VERSION}]" -qp $SCRIPTS_SRC_RPM)
    SCRIPTS_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $SCRIPTS_SRC_RPM)
    SCRIPTS_VERSION=${SCRIPTS_VER}-${SCRIPTS_REL}
fi

MPI_SELECTOR_SRC_RPM=$(/bin/ls -1 ${SRPMS}/mpi-selector-*src.rpm 2> $NULL)
if [[ -n "${MPI_SELECTOR_SRC_RPM}" && -s ${MPI_SELECTOR_SRC_RPM} ]]; then
    MPI_SELECTOR_NAME=$(rpm --queryformat "[%{NAME}]" -qp $MPI_SELECTOR_SRC_RPM)
    MPI_SELECTOR_VER=$(rpm --queryformat "[%{VERSION}]" -qp $MPI_SELECTOR_SRC_RPM)
    MPI_SELECTOR_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $MPI_SELECTOR_SRC_RPM)
    MPI_SELECTOR_VERSION=${MPI_SELECTOR_VER}-${MPI_SELECTOR_REL}
fi

MVAPICH_SRC_RPM=$(/bin/ls -1 ${SRPMS}/mvapich-*src.rpm 2> $NULL)
if [[ -n "${MVAPICH_SRC_RPM}" && -s ${MVAPICH_SRC_RPM} ]]; then
    MVAPICH_NAME=$(rpm --queryformat "[%{NAME}]" -qp $MVAPICH_SRC_RPM)
    MVAPICH_VER=$(rpm --queryformat "[%{VERSION}]" -qp $MVAPICH_SRC_RPM)
    MVAPICH_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $MVAPICH_SRC_RPM)
    MVAPICH_VERSION=${MVAPICH_VER}-${MVAPICH_REL}
fi

MVAPICH2_SRC_RPM=$(/bin/ls -1 ${SRPMS}/mvapich2-*src.rpm 2> $NULL)
if [[ -n "${MVAPICH2_SRC_RPM}" && -s ${MVAPICH2_SRC_RPM} ]]; then
    MVAPICH2_NAME=$(rpm --queryformat "[%{NAME}]" -qp $MVAPICH2_SRC_RPM)
    MVAPICH2_VER=$(rpm --queryformat "[%{VERSION}]" -qp $MVAPICH2_SRC_RPM)
    MVAPICH2_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $MVAPICH2_SRC_RPM)
    MVAPICH2_VERSION=${MVAPICH2_VER}-${MVAPICH2_REL}
fi

OPENMPI_SRC_RPM=$(/bin/ls -1 ${SRPMS}/openmpi*src.rpm 2> $NULL)
if [[ -n "${OPENMPI_SRC_RPM}" && -s ${OPENMPI_SRC_RPM} ]]; then
    OPENMPI_NAME=$(rpm --queryformat "[%{NAME}]" -qp $OPENMPI_SRC_RPM)
    OPENMPI_VER=$(rpm --queryformat "[%{VERSION}]" -qp $OPENMPI_SRC_RPM)
    OPENMPI_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $OPENMPI_SRC_RPM)
    OPENMPI_VERSION=${OPENMPI_VER}-${OPENMPI_REL}
fi
 
MPITESTS_SRC_RPM=$(/bin/ls -1 ${SRPMS}/mpitests*src.rpm 2> $NULL)
if [[ -n "${MPITESTS_SRC_RPM}" && -s ${MPITESTS_SRC_RPM} ]]; then
    MPITESTS_NAME=$(rpm --queryformat "[%{NAME}]" -qp $MPITESTS_SRC_RPM)
    MPITESTS_VER=$(rpm --queryformat "[%{VERSION}]" -qp $MPITESTS_SRC_RPM)
    MPITESTS_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $MPITESTS_SRC_RPM)
    MPITESTS_VERSION=${MPITESTS_VER}-${MPITESTS_REL}
fi

IBUTILS_SRC_RPM=$(/bin/ls -1 ${SRPMS}/ibutils*src.rpm 2> $NULL)
if [[ -n "${IBUTILS_SRC_RPM}" && -s ${IBUTILS_SRC_RPM} ]]; then
    IBUTILS_NAME=$(rpm --queryformat "[%{NAME}]" -qp $IBUTILS_SRC_RPM)
    IBUTILS_VER=$(rpm --queryformat "[%{VERSION}]" -qp $IBUTILS_SRC_RPM)
    IBUTILS_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $IBUTILS_SRC_RPM)
    IBUTILS_VERSION=${IBUTILS_VER}-${IBUTILS_REL}
fi

OPEN_ISCSI_SRC_RPM=$(/bin/ls -1 ${SRPMS}/open-iscsi*src.rpm 2> $NULL)
if [[ -n "${OPEN_ISCSI_SRC_RPM}" && -s ${OPEN_ISCSI_SRC_RPM} ]]; then
    OPEN_ISCSI_NAME=$(rpm --queryformat "[%{NAME}]" -qp $OPEN_ISCSI_SRC_RPM)
    OPEN_ISCSI_VER=$(rpm --queryformat "[%{VERSION}]" -qp $OPEN_ISCSI_SRC_RPM)
    OPEN_ISCSI_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $OPEN_ISCSI_SRC_RPM)
    OPEN_ISCSI_VERSION=${OPEN_ISCSI_VER}-${OPEN_ISCSI_REL}
    OPEN_ISCSI_SUSE_NAME="open-iscsi"
    OPEN_ISCSI_REDHAT_NAME="iscsi-initiator-utils"
    open_iscsi_rpm_name=
    case ${DISTRIBUTION} in
        SuSE)
        open_iscsi_rpm_name=${OPEN_ISCSI_SUSE_NAME}
        ;;
        redhat5)
        open_iscsi_rpm_name=${OPEN_ISCSI_REDHAT_NAME}
        ;;
    esac
fi

IB_BONDING_SRC_RPM=$(/bin/ls -1 ${SRPMS}/ib-bonding*src.rpm 2> $NULL)
if [[ -n "${IB_BONDING_SRC_RPM}" && -s ${IB_BONDING_SRC_RPM} ]]; then
    IB_BONDING_NAME=$(rpm --queryformat "[%{NAME}]" -qp $IB_BONDING_SRC_RPM)
    IB_BONDING_VER=$(rpm --queryformat "[%{VERSION}]" -qp $IB_BONDING_SRC_RPM)
    IB_BONDING_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $IB_BONDING_SRC_RPM)
    IB_BONDING_VERSION=${IB_BONDING_VER}-${IB_BONDING_REL}
fi

RDS_TOOLS_SRC_RPM=$(/bin/ls -1 ${SRPMS}/rds-tools*src.rpm 2> $NULL)
if [[ -n "${RDS_TOOLS_SRC_RPM}" && -s ${RDS_TOOLS_SRC_RPM} ]]; then
    RDS_TOOLS_NAME=$(rpm --queryformat "[%{NAME}]" -qp $RDS_TOOLS_SRC_RPM)
    RDS_TOOLS_VER=$(rpm --queryformat "[%{VERSION}]" -qp $RDS_TOOLS_SRC_RPM)
    RDS_TOOLS_REL=$(rpm --queryformat "[%{RELEASE}]" -qp $RDS_TOOLS_SRC_RPM)
    RDS_TOOLS_VERSION=${RDS_TOOLS_VER}-${RDS_TOOLS_REL}
fi

###
### Check the configuration file and include its info if it exists
###
CONFIG=${CONFIG:-"${PKG_SOURCE_DIR}/ofed.conf"}

if [ $INTERACTIVE -eq 0 ]; then
	
        if [ -s "$CONFIG" ]; then
            . $CONFIG 
        else	
            echo "$CONFIG file does not exist or empty"
            exit 1
        fi
        TT_COLOR=no
else
        touch $CONFIG || exit 1
        TT_COLOR=yes
fi

# Set network environment configuration
case ${DISTRIBUTION} in
    SuSE)
    NETWORK_CONF_DIR=${NETWORK_CONF_DIR:-"/etc/sysconfig/network"}
    ;;
    fedora*|redhat*|Rocks)
    NETWORK_CONF_DIR=${NETWORK_CONF_DIR:-"/etc/sysconfig/network-scripts"}
    ;;
    *)
    NETWORK_CONF_DIR=${NETWORK_CONF_DIR:-"/etc/infiniband"}
    ;;
esac


###
### Build Environment definition
###
BUILD_ROOT_DFT=${BUILD_ROOT:-/var/tmp/${PACKAGE}}
BUILD_ROOT=${BUILD_ROOT_DFT}
BUILD_TMP=/tmp
rpms_cache=$BUILD_TMP/rpms_installed.cache

###
### Destination path definitions
###
STACK_PREFIX=${STACK_PREFIX:-/usr}


###
### Required definitions per package
###


###
### Functions Area
###
LOG=/tmp/${PACKAGE}.${prog%.*}.$$.log

# COLOR SETTINGS
SETCOLOR_SUCCESS="echo -en \\033[1;34m"
SETCOLOR_FAILURE="echo -en \\033[1;31m"
SETCOLOR_WARNING="echo -en \\033[1;35m"
SETCOLOR_NORMAL="echo -en \\033[0;39m"

# Print message with "ERROR:" prefix
err_echo()
{
   [ "$TT_COLOR" = "yes" ] && $SETCOLOR_FAILURE
   echo -n "ERROR: $@" | tee -a $LOG
   echo >> $LOG
   [ "$TT_COLOR" = "yes" ] && $SETCOLOR_NORMAL
   echo
   return 0
}

# Print message with "WARNING:" prefix
warn_echo()
{
   [ "$TT_COLOR" = "yes" ] && $SETCOLOR_WARNING
   echo -n "WARNING: $@" | tee -a $LOG
   echo >> $LOG
   [ "$TT_COLOR" = "yes" ] && $SETCOLOR_NORMAL
   echo
   return 0
}

# Print message (bold)
pass_echo()
{
   [ "$TT_COLOR" = "yes" ] && $SETCOLOR_SUCCESS
   echo -n "$@" | tee -a $LOG
   echo >> $LOG
   [ "$TT_COLOR" = "yes" ] && $SETCOLOR_NORMAL
   echo
   return 0
}
                    

# Remove the LOG file if it exists and create a new one                    
open_log()
{
        $RM -f $LOG    
        touch $LOG
}

close_log()
{
    echo "`basename $0` finished: `date`" >> $LOG    
}

# Wait for user to press any key and continue
getch()
{
        local ans
        
        echo -n "Press Enter to continue..."
        read ans
        return 0
}


# Clean the temporary files upon failure or after cancelling installation
cleanup()
{
        local ACTION=""
        local ans=""

        if [ $RUNNING_CLEANUP -eq 1 ]; then
            return 0
        fi
        
        RUNNING_CLEANUP=1
        
        if [ $INTERACTIVE -eq 1 ]; then
                # IF in install mode run uninstall
                if [ "$prog" == "install.sh" ]; then
                        echo
                        read -p  "Do you want to remove the already installed ${PACKAGE} packages? [Y/n]:" ans
                        if ! [[ "$ans" == "" || "$ans" == "y" || "$ans" == "Y" || "$ans" == "yes" ]]; then
                                return 0
                        fi   
                        ACTION="uninstall"
                else
                        # IF in build mode delete BUILD_ROOT
                        echo
                        if [ -d "$BUILD_ROOT" ]; then
                                read -p  "Do you want to remove the temporary build directory ${BUILD_ROOT}? [Y/n]:" ans
                                if ! [[ "$ans" == "" || "$ans" == "y" || "$ans" == "Y" || "$ans" == "yes" ]]; then
                                        return 0
                                fi   
                        fi
                        ACTION="delete"
                fi
        else # Non interactive mode
                if [ "$prog" == "install.sh" ]; then
                        ACTION="leave" # ACTION="uninstall"
                else
                        ACTION="leave"
                fi
        fi
        
        case "$ACTION" in
              "uninstall")
                      rmInstalled
              ;;
              "delete")
                      [ -d "$BUILD_ROOT" ] && $RM -rf $BUILD_ROOT
              ;;
              *)
              ;;
        esac
        
        if [ $INTERACTIVE -eq 1 ]; then
                getch
        fi
        
}

# Returns "0" if $1 is integer and "1" otherwise
is_integer() 
{
        printf "%s\n" $1 |grep -E "^[+-]?[0-9]+$" > $NULL
        return $?
}

# Stop running propeller
kill_propel()
{
        [ -z $PR_PID ] && return 0
        if ! is_integer $PR_PID; then
                return 0
        fi
        if ! ( ps -p $PR_PID > $NULL 2>&1 ); then
                return 0
        fi
        
        kill -15 $PR_PID  > $NULL 2>&1
        sleep 1
}

trap_handler()
{
        kill_propel
        exit 1

}

# Get user's trap (Ctrl-C or kill signals)
trap 'trap_handler' 2 9 15

# Execute the command $@ and check exit status
ex()
{
        EXECUTING=1

        perl -e 'exit((-t STDOUT) ? 0:1)'
        if [ $? -eq 0 ]; then
            $PROPEL &
            PR_PID=$!
        fi
     
        echo Running $@
        eval "$@" >> $LOG 2>&1
        if [ $? -ne 0 ]; then
             echo
             err_echo Failed executing \"$@\"
             echo
             cd ${curr_dir}
             echo See log file: $LOG
             echo
     
             kill_propel
     
             if [ $INTERACTIVE -eq 1 ]; then
                getch
             fi
             cleanup
             exit 5
        fi
     
        kill_propel
        
        EXECUTING=0   
}

# Execute the command $@ in silent mode and check exit status
ex_silent()
{
        EXECUTING=1
        
        #   echo Running $@
        eval "$@" >> $LOG 2>&1
        if [ $? -ne 0 ]; then
             echo
             err_echo Failed in execution \"$@\"
             echo
             cd ${curr_dir}
             echo See log file: $LOG
             echo
        
             if [ $INTERACTIVE -eq 1 ]; then
                getch
             fi
             cleanup
             exit 5
        fi
        
        EXECUTING=0   
}

# Check that "root" is the current user
check_user()
{
        local user
        user=`/usr/bin/id -u`
        if [ $user -ne 0 ] && [ "$prog" == "install.sh" ]; then
            err_echo "You must be root to run this install program"
            echo
            echo "Note: In order to build RPMs as $(whoami) run ./build.sh script"
            echo
            exit 1
        fi
}

# Check input from user (default is "No")
get_ans()
{
read ans
case $ans in
    Y|YES|y|yes)
        ans=""
        return 0
    ;;    
    *)
        ans=""
        return 1    
    ;;
esac        
}

# Show README or the ${PACKAGE} Installation guide file
view_readme()
{
        local prog=`basename $0`
        
        if [ "$prog" == "install.sh" ]; then
                if [ -e $PKG_SOURCE_DIR/docs/${PACKAGE}_Installation_Guide.txt ]; then
                    less $PKG_SOURCE_DIR/docs/${PACKAGE}_Installation_Guide.txt
                else
                    warn_echo $PKG_SOURCE_DIR/docs/${PACKAGE}_Installation_Guide.txt does not exist
                    getch
                fi                        
        else
                if [ -e $PKG_SOURCE_DIR/README.txt ]; then
                    less $PKG_SOURCE_DIR/README.txt
                else
                    warn_echo $PKG_SOURCE_DIR/README.txt does not exist
                    getch
                fi                        
        	
        fi
        
        return 0
}

# Check disk space requirments $1 - required size $2 - directory to check
check_space_req()
{
        local space_req=$1
        local dir=$2
        
        shift 2
        
        while [ ! -d $dir ]
        do
                dir=${dir%*/*}
        done
        
        local avail_space=`$DF $dir | tail -1 | awk '{print$4}' | tr -d '[:space:]'`
        
        if ! is_integer $avail_space; then
                # Wrong avail_space found
                return 0
        fi
        
        if [ $avail_space -lt $space_req ]; then
                echo
                err_echo "Not enough disk space in the ${dir} directory. Required ${space_req}KB"
                echo
                if [ $INTERACTIVE -eq 1 ]; then
                    getch
                fi
                exit 1
        fi

        return 0
}

# Compare package versions $1 and $2. Return 0 if $1=$2, 1 if $1 > $2, 2 - if $1 < $2
compare()
{
        local ver1=$(echo $1 | tr -d '[:alpha:]')
        local ver2=$(echo $2 | tr -d '[:alpha:]')
        local dots1=0
        local dots2=0
        local dots=0
        local diff=0
        
        if [ "$ver1" == "$ver2" ]; then
            return 0
        fi
        
        [ "$ver1" == "" ] && ver1=0
        [ "$ver2" == "" ] && ver2=0
        
        dots1=$(echo $ver1 | tr -d '[:digit:]' | tr -d '[:space:]' | wc -c | tr -d '[:space:]')
        dots2=$(echo $ver2 | tr -d '[:digit:]' | tr -d '[:space:]' | wc -c | tr -d '[:space:]')
        
        if [ $dots1 -gt $dots2 ]; then
            dots=$dots2
            diff=1
        elif [ $dots1 -lt $dots2 ]; then
            dots=$dots1
            diff=2
        else
            dots=$dots1
            diff=0    
        fi
        
        let dots++
        for (( i=1 ; i <= $dots ; i++ ))
        do
            [ $(echo $ver1 | cut -d . -f $i | tr -d '[:space:]') -gt $(echo $ver2 | cut -d . -f $i | tr -d '[:space:]') ] && return 1
            [ $(echo $ver1 | cut -d . -f $i | tr -d '[:space:]') -lt $(echo $ver2 | cut -d . -f $i | tr -d '[:space:]') ] && return 2
        done
        
        [ $diff -eq 1 ] && return 1
        [ $diff -eq 2 ] && return 2
        
        return 3
}

count_ib_ports()
{
    local cnt=0
    local tmp_cnt=0
    
    tmp_cnt=$(/sbin/lspci -n 2> $NULL | grep "15b3:6282" | wc -l | tr -d '[:space:]') # InfiniHost III Ex mode
    cnt=$[ $cnt + 2*${tmp_cnt} ]
    
    tmp_cnt=$(/sbin/lspci -n 2> $NULL | grep -E "15b3:5e8c|15b3:6274" | wc -l | tr -d '[:space:]') # InfiniHost III Lx mode
    cnt=$[ $cnt + ${tmp_cnt} ]

    tmp_cnt=$(/sbin/lspci -n 2> $NULL | grep -E "15b3:5a44|15b3:6278" | wc -l | tr -d '[:space:]') # InfiniHost mode
    cnt=$[ $cnt + 2*${tmp_cnt} ]

    # TBD insert count for ipath devices
    
    return $cnt
}

print_package_info()
{
    local package=$1
    shift
    local DESCR=$($RPM --queryformat "[%{DESCRIPTION}]" -q $package)
    local VERSION=$($RPM --queryformat "[%{VERSION}]" -q $package)
    echo
    echo "NAME: $package"
    echo "VERSION: $VERSION"
    echo "Description: $DESCR"
}

# View installed packages
showInstalled()
{
        local RC=0
        local mpi_packs=""
        local package=""

        for package in $ALL_PACKAGES $PREV_RELEASE_PACKAGES
        do
            case $package in
                mvapich)
                    mpi_packs=$($RPM -qa | grep ${MVAPICH_NAME})
                    if [ -n "${mpi_packs}" ]; then
                        for mpi_pack in ${mpi_packs}
                        do
                            if ( $RPM -q $mpi_pack > $NULL 2>&1 ); then
                                print_package_info $mpi_pack
                                let RC++
                            fi
                        done
                    fi
                ;;
                mvapich2)
                    mpi_packs=$($RPM -qa | grep ${MVAPICH2_NAME})
                    if [ -n "${mpi_packs}" ]; then
                        for mpi_pack in ${mpi_packs}
                        do
                            if ( $RPM -q $mpi_pack > $NULL 2>&1 ); then
                                print_package_info $mpi_pack
                                let RC++
                            fi
                        done
                    fi
                ;;
                openmpi)
                    mpi_packs=$($RPM -qa | grep ${OPENMPI_NAME})
                    if [ -n "${mpi_packs}" ]; then
                        for mpi_pack in ${mpi_packs}
                        do
                            if ( $RPM -q $mpi_pack > $NULL 2>&1 ); then
                                print_package_info $mpi_pack
                                let RC++
                            fi
                        done
                    fi
                ;;
                *)
                    if ( $RPM -q $package > $NULL 2>&1 ); then
                        print_package_info $package
                        let RC++
                    fi
                ;;
            esac
        done    
        
        if [ $RC -eq 0 ]; then
            echo
            echo    There are no ${PACKAGE} Software Installed
            echo
        fi
  
        return 0
}

# Check if the required compiler - $1 exist
# In case of the Open MPI package, the Fortran compiler is not required
is_compiler()
{
    local compiler=$1
    local mpipackage=$2
    shift 2

    case ${compiler} in
        gcc)
            is_gpp=0
            is_gcc_g77=0
            is_gfortran=0
            if ! ( $WHICH gcc > $NULL 2>&1 ); then
                return 1
            fi
            if [ "${mpipackage}" == "mvapich" ] || [ "${mpipackage}" == "mvapich2" ]; then
                if ( $WHICH gfortran > $NULL 2>&1 ); then
                    is_gfortran=1
                    return 0
                elif ( $WHICH g77 > $NULL 2>&1 ); then
                    is_gcc_g77=1
                    return 0
                else
                    return 1
                fi
            else
                if ( $WHICH g++ > $NULL 2>&1 ); then
                    is_gpp=1
                fi
                if ( $WHICH g77 > $NULL 2>&1 ); then
                    is_gcc_g77=1
                fi
                if ( $WHICH gfortran > $NULL 2>&1 ); then
                    is_gfortran=1
                fi
                return 0
            fi
        ;;
        pathscale)
            is_pathCC=0
            is_pathf90=0
            if ! ( $WHICH pathcc > $NULL 2>&1 ); then
                return 1
            fi
            if [ "${mpipackage}" == "mvapich" ] || [ "${mpipackage}" == "mvapich2" ]; then
                if ( $WHICH pathCC > $NULL 2>&1 && $WHICH pathf90 > $NULL 2>&1 ); then
                    is_pathCC=1
                    is_pathf90=1
                    return 0
                else
                    return 1
                fi
            else
                if ( $WHICH pathCC > $NULL 2>&1 ); then
                    is_pathCC=1
                fi
                if ( $WHICH pathf90 > $NULL 2>&1 ); then
                    is_pathf90=1
                fi
                return 0
            fi
        ;;
        pgi)
            is_pgCC=0
            is_pgf77=0
            is_pgf90=0
            if ! ( $WHICH pgcc > $NULL 2>&1 ); then
                return 1
            fi
            if [ "${mpipackage}" == "mvapich" ] || [ "${mpipackage}" == "mvapich2" ]; then
                if ( $WHICH pgf77 > $NULL 2>&1 && $WHICH pgf90 > $NULL 2>&1 ); then
                    is_pgf77=1
                    is_pgf90=1
                    return 0
                else
                    return 1
                fi
            else
                if ( $WHICH pgCC > $NULL 2>&1 ); then
                    is_pgCC=1
                fi
                if ( $WHICH pgf77 > $NULL 2>&1 ); then
                    is_pgf77=1
                fi
                if ( $WHICH pgf90 > $NULL 2>&1 ); then
                    is_pgf90=1
                fi
                return 0
            fi
        ;;
        intel)
            is_icpc=0
            is_ifort=0
            if ! ( $WHICH icc > $NULL 2>&1 ); then
                return 1
            fi
            if [ "${mpipackage}" == "mvapich" ] || [ "${mpipackage}" == "mvapich2" ]; then
                if ( $WHICH icpc > $NULL 2>&1 && $WHICH ifort > $NULL 2>&1 ); then
                    is_icpc=1
                    is_ifort=1
                    return 0
                else
                    return 1
                fi
            else
                if ( $WHICH icpc > $NULL 2>&1 ); then
                    is_icpc=1
                fi
                if ( $WHICH ifort > $NULL 2>&1 ); then
                    is_ifort=1
                fi
                return 0
            fi
        ;;
        *)
            warn_echo "This compiler is not supported"
            return 1
        ;;
    esac
    
}

# Prompt for MVAPICH2 build options.
mvapich2_config() {
    local choice=""
    local blcr

    # Set default values if not in interactive mode and no values were
    # were stored in the configuration file.
    if [ -z "$MVAPICH2_CONF_done" ] && [ $INTERACTIVE = 0 ]; then
        MVAPICH2_CONF_impl=ofa

        if ! ( grep -w MVAPICH2_CONF_impl $CONFIG > $NULL 2>&1 ); then
            echo "MVAPICH2_CONF_impl=\"${MVAPICH2_CONF_impl}\"" >> $CONFIG
        fi

        MVAPICH2_CONF_romio=1

        if ! ( grep -w MVAPICH2_CONF_romio $CONFIG > $NULL 2>&1 ); then
            echo "MVAPICH2_CONF_romio=\"${MVAPICH2_CONF_romio}\"" >> $CONFIG
        fi

        MVAPICH2_CONF_shared_libs=1

        if ! ( grep -w MVAPICH2_CONF_shared_libs $CONFIG > $NULL 2>&1 ); then
            echo "MVAPICH2_CONF_shared_libs=\"${MVAPICH2_CONF_shared_libs}\"" >> $CONFIG
        fi

        MVAPICH2_CONF_multithread=0

        if ! ( grep -w MVAPICH2_CONF_multithread $CONFIG > $NULL 2>&1 ); then
            echo "MVAPICH2_CONF_multithread=\"${MVAPICH2_CONF_multithread}\"" >> $CONFIG
        fi

        MVAPICH2_CONF_ckpt=0

        if ! ( grep -w MVAPICH2_CONF_ckpt $CONFIG > $NULL 2>&1 ); then
            echo "MVAPICH2_CONF_ckpt=\"${MVAPICH2_CONF_ckpt}\"" >> $CONFIG
        fi

        MVAPICH2_CONF_done=1

        if ! ( grep -w MVAPICH2_CONF_done $CONFIG > $NULL 2>&1 ); then
            echo "MVAPICH2_CONF_done=\"${MVAPICH2_CONF_done}\"" >> $CONFIG
        fi

        return
    elif [ "$MVAPICH2_CONF_done" = 1 ]; then
        return
    fi

    cat <<EOF

Please choose an implementation of MVAPICH2:

1) OFA (IB and iWARP)
2) uDAPL

EOF
    while [ -z "$choice" ] || [[ $choice != [0-9] ]] || [ $choice -lt 1 ] || [ $choice -gt 2 ]; do
        read -p "Implementation [1]: "
        choice=${REPLY:-1}
    done

    if [ $choice -eq 1 ]; then
        MVAPICH2_CONF_impl=ofa
    elif [ $choice -eq 2 ]; then
        MVAPICH2_CONF_impl=udapl
    fi

    if ! ( grep -w MVAPICH2_CONF_impl $CONFIG > $NULL 2>&1 ); then
        echo "MVAPICH2_CONF_impl=\"${MVAPICH2_CONF_impl}\"" >> $CONFIG
    fi

    while [ -z "$MVAPICH2_CONF_romio" ]; do
        read -p "Enable ROMIO support [Y/n]: " choice

        if [ -z "$choice" ] || [[ $choice == [yY] ]] || [[ $choice == [yY][eE][sS] ]]; then
            MVAPICH2_CONF_romio=1
        elif [[ $choice == [nN] ]] || [[ $choice == [nN][oO] ]]; then
            MVAPICH2_CONF_romio=0
        fi
    done

    if ! ( grep -w MVAPICH2_CONF_romio $CONFIG > $NULL 2>&1 ); then
        echo "MVAPICH2_CONF_romio=\"${MVAPICH2_CONF_romio}\"" >> $CONFIG
    fi

    while [ -z "$MVAPICH2_CONF_shared_libs" ]; do
        read -p "Enable shared library support [Y/n]: " choice

        if [ -z "$choice" ] || [[ $choice == [yY] ]] || [[ $choice == [yY][eE][sS] ]]; then
            MVAPICH2_CONF_shared_libs=1
        elif [[ $choice == [nN] ]] || [[ $choice == [nN][oO] ]]; then
            MVAPICH2_CONF_shared_libs=0
        fi
    done

    if ! ( grep -w MVAPICH2_CONF_shared_libs $CONFIG > $NULL 2>&1 ); then
        echo "MVAPICH2_CONF_shared_libs=\"${MVAPICH2_CONF_shared_libs}\"" >> $CONFIG
    fi

    cat <<EOF
Multithread support should only be enabled only if thread safety is required.
There may be a slight performance penalty for single threaded only use.
EOF

    while [ -z "$MVAPICH2_CONF_multithread" ]; do
        read -p "Enable multithread support [y/N]: " choice

        if [ -z "$choice" ] || [[ $choice == [nN] ]] || [[ $choice == [nN][oO] ]]; then
            MVAPICH2_CONF_multithread=0
        elif [[ $choice == [yY] ]] || [[ $choice == [yY][eE][sS] ]]; then
            MVAPICH2_CONF_multithread=1
        fi
    done

    if ! ( grep -w MVAPICH2_CONF_multithread $CONFIG > $NULL 2>&1 ); then
        echo "MVAPICH2_CONF_multithread=\"${MVAPICH2_CONF_multithread}\"" >> $CONFIG
    fi

    # OFA specific options.
    if [ $MVAPICH2_CONF_impl = "ofa" ] && [ $MVAPICH2_CONF_multithread -eq 0 ]; then
        choice=0

        while [ $choice = 0 ]; do
            read -p "Enable Checkpoint-Restart support [y/N]: " choice

            if [ -z "$choice" ] || [[ $choice == [nN] ]] || [[ $choice == [nN][oO] ]]; then
                MVAPICH2_CONF_ckpt=0
                choice=1
            elif [[ $choice == [yY] ]] || [[ $choice == [yY][eE][sS] ]]; then
                read -p "BLCR installation directory [or nothing if not installed]: " blcr

                if [ -d "$blcr" ]; then
                    MVAPICH2_CONF_ckpt=1
                    MVAPICH2_CONF_blcr_home="$blcr"
                    choice=1
                else
                    echo "BLCR installation directory not found."
                    choice=0
                fi
            else
                choice=0
            fi
        done
    else
        MVAPICH2_CONF_ckpt=0
    fi

    if [ $MVAPICH2_CONF_impl = "ofa" ]; then
        if ! ( grep -w MVAPICH2_CONF_ckpt $CONFIG > $NULL 2>&1 ); then
            echo "MVAPICH2_CONF_ckpt=\"${MVAPICH2_CONF_ckpt}\"" >> $CONFIG
        fi

        if [ $MVAPICH2_CONF_ckpt -eq 1 ]; then
            if ! ( grep -w MVAPICH2_CONF_blcr_home $CONFIG > $NULL 2>&1 ); then
                echo "MVAPICH2_CONF_blcr_home=\"${MVAPICH2_CONF_blcr_home}\"" >> $CONFIG
            fi
        fi
    fi

    # uDAPL specific options.
    if [ $MVAPICH2_CONF_impl = "udapl" ]; then
        cat <<EOF

Cluster size:

1) Small
2) Medium
3) Large

EOF
        choice=""

        while [ -z "$choice" ] || [[ $choice != [0-9] ]] || [ $choice -lt 1 ] || [ $choice -gt 3 ]; do
            read -p "Cluster size [1]: "
            choice=${REPLY:-1}
        done

        if [ $choice -eq 1 ]; then
            MVAPICH2_CONF_vcluster=small
        elif [ $choice -eq 2 ]; then
            MVAPICH2_CONF_vcluster=medium
        elif [ $choice -eq 3 ]; then
            MVAPICH2_CONF_vcluster=large
        fi

        if ! ( grep -w MVAPICH2_CONF_vcluster $CONFIG > $NULL 2>&1 ); then
            echo "MVAPICH2_CONF_vcluster=\"${MVAPICH2_CONF_vcluster}\"" >> $CONFIG
        fi

        cat <<EOF

I/O Bus:

1) PCI-Express
2) PCI-X

EOF
        choice=""

        while [ -z "$choice" ] || [[ $choice != [0-9] ]] || [ $choice -lt 1 ] || [ $choice -gt 2 ]; do
            read -p "I/O Bus [1]: "
            choice=${REPLY:-1}
        done

        if [ $choice -eq 1 ]; then
            MVAPICH2_CONF_io_bus=pci-ex
        elif [ $choice -eq 2 ]; then
            MVAPICH2_CONF_io_bus=pci-x
        fi

        if ! ( grep -w MVAPICH2_CONF_io_bus $CONFIG > $NULL 2>&1 ); then
            echo "MVAPICH2_CONF_io_bus=\"${MVAPICH2_CONF_io_bus}\"" >> $CONFIG
        fi

        if [ $MVAPICH2_CONF_io_bus = "pci-ex" ]; then
            cat <<EOF

Link Speed:

1) SDR
2) DDR

EOF
            choice=""

            while [ -z "$choice" ] || [[ $choice != [0-9] ]] || [ $choice -lt 1 ] || [ $choice -gt 2 ]; do
                read -p "Link Speed [1]: "
                choice=${REPLY:-1}
            done

            if [ $choice -eq 1 ]; then
                MVAPICH2_CONF_link_speed=sdr
            elif [ $choice -eq 2 ]; then
                MVAPICH2_CONF_link_speed=ddr
            fi
        else
            MVAPICH2_CONF_link_speed=sdr
        fi

        if ! ( grep -w MVAPICH2_CONF_link_speed $CONFIG > $NULL 2>&1 ); then
            echo "MVAPICH2_CONF_link_speed=\"${MVAPICH2_CONF_link_speed}\"" >> $CONFIG
        fi

        read -p "Default DAPL provider [ib0]: "
        MVAPICH2_CONF_dapl_provider=${REPLY:-ib0}

        if ! ( grep -w MVAPICH2_CONF_dapl_provider $CONFIG > $NULL 2>&1 ); then
            echo "MVAPICH2_CONF_dapl_provider=\"${MVAPICH2_CONF_dapl_provider}\"" >> $CONFIG
        fi
    fi

    MVAPICH2_CONF_done=1

    if ! ( grep -w MVAPICH2_CONF_done $CONFIG > $NULL 2>&1 ); then
        echo "MVAPICH2_CONF_done=\"${MVAPICH2_CONF_done}\"" >> $CONFIG
    fi
}

check_open_iscsi()
{

    if [ ! -z "${open_iscsi_rpm_name}" ]; then
        if ( $RPM -q ${open_iscsi_rpm_name} > $NULL 2>&1 ) \
            && ! ( $RPM --queryformat "[%{VENDOR}]" -q ${open_iscsi_rpm_name} | grep -i Voltaire > $NULL 2>&1 ); then
                if [ $INTERACTIVE -eq 1 ]; then
                        pass_echo "In order to install iSER ${open_iscsi_rpm_name} package should be upgraded."
                        choice=""
                        read -p "Do you want to upgrade ${open_iscsi_rpm_name}? [y/N]: " choice
                        if [[ $choice == [yY] ]] || [[ $choice == [yY][eE][sS] ]]; then
                                upgrade_open_iscsi=yes 
                                if ! ( grep -w upgrade_open_iscsi $CONFIG > $NULL 2>&1 ); then
                                        echo "upgrade_open_iscsi=${upgrade_open_iscsi}" >> $CONFIG
                                fi
                        else
                                err_echo "Please uninstall ${open_iscsi_rpm_name} before installing ${PACKAGE} with iSER support."
                                exit 1
                        fi
                else
                        if [ "X${upgrade_open_iscsi}" != "Xyes" ]; then
                                err_echo "Please uninstall ${open_iscsi_rpm_name} before installing ${PACKAGE} with iSER support."
                                echo "  Or put \"upgrade_open_iscsi=yes\" in the $CONFIG"
                                exit 1
                        fi
                fi
        fi
    fi
}

# Set Compilation environment for MPI
set_mpi_env()
{

        local mpipackage=$1
        shift
        local COMPILERS_FOUND=""
        local EXCLUDED_COMPILERS=""
        local mpi_compiler=""
        local ans=""

        if [ $INTERACTIVE -eq 1 ]; then

                for mpi_compiler in ${MPI_SUPPORTED_COMPILERS}
                do
                    if is_compiler ${mpi_compiler} ${mpipackage}; then
                        COMPILERS_FOUND="$COMPILERS_FOUND ${mpi_compiler}"
                    fi
                done

            # COMPILERS_FOUND=$(echo $COMPILERS_FOUND | tr -s ' ' '\n' | sort -rn | uniq | tr -s '\n' ' ')
            if [ -n "$COMPILERS_FOUND" ]; then
                echo
                echo "The following compiler(s) on your system can be used to build/install ${mpipackage}: $COMPILERS_FOUND"
                echo
            fi


            printed_msg0=${printed_msg0:-0}
            if [ $printed_msg0 -eq 0 ]; then
                if [ $(echo -n ${COMPILERS_FOUND} | wc -w) -gt 1 ]; then
                    echo "Next you will be prompted to choose the compiler(s) with which to build/install the MPI RPM(s)"
                    echo
                fi
                printed_msg0=1
            fi

            MPI_COMPILER=""

            for mpi_compiler in $COMPILERS_FOUND
            do
                read -p "Do you wish to create/install an ${mpipackage} RPM with ${mpi_compiler}? [Y/n]:" ans
                if [[ "$ans" == "" || "$ans" == "y" || "$ans" == "Y" || "$ans" == "yes" ]]; then
                    MPI_COMPILER="$MPI_COMPILER ${mpi_compiler}"
                fi
            done
        else # Unattended mode
            case ${mpipackage} in
                mvapich)
                    if [ ! -z "${MPI_COMPILER_mvapich}" ]; then
                        MPI_COMPILER=${MPI_COMPILER_mvapich}
                    fi
                ;;
                mvapich2)
                    if [ ! -z "${MPI_COMPILER_mvapich2}" ]; then
                        MPI_COMPILER=${MPI_COMPILER_mvapich2}
                    fi
                ;;
                openmpi)
                    if [ ! -z "${MPI_COMPILER_openmpi}" ]; then
                        MPI_COMPILER=${MPI_COMPILER_openmpi}
                    fi
                ;;
            esac

            if [ ! -n "${MPI_COMPILER}" ]; then
                MPI_COMPILER=gcc
                echo "The MPI_COMPILER_${mpipackage} variable is not defined. Trying the default compiler: ${MPI_COMPILER}"
            fi
            for mpi_compiler in ${MPI_COMPILER}
            do
                if ! is_compiler ${mpi_compiler} ${mpipackage}; then
                    if [ "${mpi_compiler}" == "gcc" ] && [ "${mpipackage}" == "mvapich" ]; then
                        mpi_compiler="gcc-g77"
                    fi
                    EXCLUDED_COMPILERS="$EXCLUDED_COMPILERS ${mpi_compiler}"
                else
                    COMPILERS_FOUND="$COMPILERS_FOUND ${mpi_compiler}"
                fi
            done

            MPI_COMPILER=$(echo $COMPILERS_FOUND | tr -s ' ' '\n' | sort -rn | uniq | tr -s '\n' ' ')

            if [ -n "${EXCLUDED_COMPILERS}" ]; then
                warn_echo "The following compiler(s) for ${mpipackage} were not found on the system: ${EXCLUDED_COMPILERS}"
            fi

        fi


        case ${mpipackage} in
            mvapich)
                if [ ! -n "${MPI_COMPILER}" ]; then
                    warn_echo "No compilers for ${mpipackage} were found"
                    return 1
                fi

                MPI_COMPILER_mvapich=${MPI_COMPILER}
                if ! ( grep -w MPI_COMPILER_mvapich $CONFIG > $NULL 2>&1 ); then
                    echo "MPI_COMPILER_mvapich=\"${MPI_COMPILER_mvapich}\"" >> $CONFIG
                fi
                echo
                echo "The following compiler(s) will be used to ${prog%*.*} the ${mpipackage} RPM(s): $MPI_COMPILER_mvapich"
            ;;
            mvapich2)
                if [ ! -n "${MPI_COMPILER}" ]; then
                    warn_echo "No compilers for ${mpipackage} were found"
                    return 1
                fi
                MPI_COMPILER_mvapich2=${MPI_COMPILER}
                if ! ( grep -w MPI_COMPILER_mvapich2 $CONFIG > $NULL 2>&1 ); then
                    echo "MPI_COMPILER_mvapich2=\"${MPI_COMPILER_mvapich2}\"" >> $CONFIG
                fi
                echo
                echo "The following compiler(s) will be used to ${prog%*.*} the ${mpipackage} RPM(s): $MPI_COMPILER_mvapich2"
                # MVAPICH2 can be built with many options.  The configuration
                # function below asks the user how to build, and it only will
                # do so if the configuration values have not already been
                # read from the $CONFIG file.
                mvapich2_config
            ;;
            openmpi)
                if [ ! -n "${MPI_COMPILER}" ]; then
                    warn_echo "No compilers for ${mpipackage} were found"
                    return 1
                fi
                MPI_COMPILER_openmpi=${MPI_COMPILER}
                if ! ( grep -w MPI_COMPILER_openmpi $CONFIG > $NULL 2>&1 ); then
                    echo "MPI_COMPILER_openmpi=\"${MPI_COMPILER_openmpi}\"" >> $CONFIG
                fi
                echo
                echo "The following compiler(s) will be used to ${prog%*.*} the ${mpipackage} RPM(s): $MPI_COMPILER_openmpi"
            ;;
        esac
        MPI_COMPILER=""
        return 0
}

# Select the packages to install
select_packages()
{

        if [ $INTERACTIVE -eq 1 ]; then

                local package=""
                local package_var=""
                local ans_select=""
                # Configuration parameters initialization
                touch $CONFIG

                # Save previous configuration file
                [ -s $CONFIG ] && $MV -f $CONFIG ${CONFIG}.save

                cat $NULL > $CONFIG

                echo STACK_PREFIX=$STACK_PREFIX >> $CONFIG
                echo BUILD_ROOT=$BUILD_ROOT >> $CONFIG

                show_menu "select_mode_menu"

                read ans_select
                case $ans_select in
                        $BASIC)
                                SELECTED_PACKAGES="$BASIC_PACKAGES $BASIC_IB_KERNEL_PACKAGES"
                                for package in $SELECTED_PACKAGES
                                do
                                        package_var=$(echo -n ${package} | sed -e 's/-/_/g')
                                        echo "${package_var}=y" >> $CONFIG
                                done
                        ;;
                        $HPC)
                                SELECTED_PACKAGES="$HPC_PACKAGES $IB_KERNEL_PACKAGES"
                                for package in $SELECTED_PACKAGES 
                                do
                                        package_var=$(echo -n ${package} | sed -e 's/-/_/g')
                                        echo "${package_var}=y" >> $CONFIG
                                done
                        ;;
                        $ALL)
                                SELECTED_PACKAGES="$ALL_PACKAGES $IB_KERNEL_PACKAGES"
                                for package in $SELECTED_PACKAGES
                                do
                                        package_var=$(echo -n ${package} | sed -e 's/-/_/g')
                                        echo "${package_var}=y" >> $CONFIG
                                done
                        ;;
                        $CUSTOM)
                                echo
                                echo "RPM packages:"
                                for package in $ALL_PACKAGES
                                do
                                        pack_descr=$(eval echo \$\{${package}_descr\})
                                        package_var=$(echo -n ${package} | sed -e 's/-/_/g')
                                        if [ "${package}" == "libibverbs" ]; then
                                            echo "User level libraries/applications:"
                                        fi
                                        echo -n "Install ${package}: [y/N]:"

                                        if get_ans; then
                                            if [ "$SELECTED_PACKAGES" == "" ]; then
                                                SELECTED_PACKAGES="${package}"
                                            else
                                                SELECTED_PACKAGES="$SELECTED_PACKAGES ${package}"
                                            fi
                                            echo "${package_var}=y" >> $CONFIG
                                            if [ "${package}" == "kernel-ib" ]; then
                                                echo "Kernel level modules:"
                                                for package in $IB_KERNEL_PACKAGES
                                                do
                                                        pack_descr=$(eval echo \$\{${package}_descr\})
                                                        package_var=$(echo -n ${package} | sed -e 's/-/_/g')

                                                        if [ "${package}" == "ib_iser" ]; then
                                                            echo -n "Install ${package} (open-iscsi will also be installed): [y/N]:"
                                                        else
                                                            echo -n "Install ${package}: [y/N]:"
                                                        fi

                                                        if get_ans; then
                                                            if [ "$SELECTED_PACKAGES" == "" ]; then
                                                                SELECTED_PACKAGES="${package}"
                                                            else
                                                                SELECTED_PACKAGES="$SELECTED_PACKAGES ${package}"
                                                            fi
                                                            echo "${package_var}=y" >> $CONFIG
                                                        else
                                                            echo "${package_var}=n" >> $CONFIG
                                                        fi
                                                done
                                            fi
                                        else
                                            echo "${package_var}=n" >> $CONFIG
                                        fi
                                done
                        ;;
                        q|Q)
                                # Exit from install/build script
                                exit 0
                        ;;
                        *)
                                select_packages
                        ;;
                esac
                . $CONFIG
        else # Unatended mode
            # Read configuration from CONFIG file
            # . $CONFIG
            SELECTED_PACKAGES=$(perl -n -e 'if(m@^\s*#.*@) { $filter = 1; } elsif (m@^\s*ib_.*@) { $filter = 1; } else { $filter = 0; }' -e 'if(m@^\s*(.*)=y\s*$@ and $filter == 0) { print "$1 " }' $CONFIG)
            SELECTED_OFA_KERNEL_PACKAGES=$(perl -n -e 'if(m@^\s*#.*@) { $filter = 1; } else { $filter = 0; }' -e 'if(m@^\s*(ib_.*)=y@ and $filter == 0) { print "$1 " }' $CONFIG)
            SELECTED_PACKAGES="$SELECTED_OFA_KERNEL_PACKAGES $SELECTED_PACKAGES"
            # echo "DEBUG: SELECTED_PACKAGES=$SELECTED_PACKAGES"
        fi
}

# Display installation/build menus
show_menu()
{

    clear
     case "$1" in
                 "main_menu")
cat << EOF_INST

         ${PACKAGE} Distribution Software Installation Menu

          1) View ${PACKAGE} Installation Guide
          2) Install ${PACKAGE} Software 
          3) Show Installed Software
          4) Configure IPoIB Network Interface and OpenSM Server
          5) Uninstall ${PACKAGE} Software
          6) Build ${PACKAGE} Software RPMs

          Q) Exit

EOF_INST
          
    echo -n   "Select Option [1-6]:"
                         ;;
                 "build_menu")
cat << EOF_BUILD

         ${PACKAGE} Software Build Menu

          1) View README (read "Building ${PACKAGE} SW RPM packages")
          2) Build ${PACKAGE} Software RPMs

          Q) Exit

EOF_BUILD

    echo -n   "Select Option [1-2]:"

                 ;;
                 "select_mode_menu")
cat << EOF_MODE_MENU

          Select ${PACKAGE} Software

          1) Basic (${PACKAGE} modules and basic user level libraries)
          2) HPC (${PACKAGE} modules and libraries, MPI and diagnostic tools)
          3) All packages (all of Basic, HPC)
          4) Customize

          Q) Exit

EOF_MODE_MENU

    echo -n   "Select Option [1-4]:"

                 ;;
                 *)
                     return 1
                 ;;
     esac
    return 0
}


OFA_DEST=${STACK_PREFIX}
OPENSM_DEST=${STACK_PREFIX}
OSU_DEST=${STACK_PREFIX}/mpi/osu
SCRIPTS_DEST=${STACK_PREFIX}/scripts
FW_DEST=${STACK_PREFIX}

###
### Required definitions per package
###

kernel_ib_RUN_REQ=""
kernel_ib_devel_RUN_REQ=""
libibverbs_RUN_REQ="gcc_3.3.3"
libibverbs_devel_RUN_REQ=""
libibverbs_utils_RUN_REQ=""
libibcm_RUN_REQ=""
libmthca_RUN_REQ=""
libmthca_devel_RUN_REQ=""
libehca_RUN_REQ=""
libehca_devel_RUN_REQ=""
libipathverbs_RUN_REQ=""
libipathverbs_devel_RUN_REQ=""
libsdp_RUN_REQ=""
srptools_RUN_REQ=""
ipoibtools_RUN_REQ=""
tvflash_RUN_REQ=""
libibcommon_RUN_REQ=""
libibmad_RUN_REQ=""
libibumad_RUN_REQ=""
openib_diags_RUN_REQ=""
ibutils_RUN_REQ="tcl_8.4 tcl-devel_8.4 tk"
opensm_RUN_REQ=""
libopensm_RUN_REQ=""
libopensm_devel_RUN_REQ=""
libosmcomp_RUN_REQ=""
libosmcomp_devel_RUN_REQ=""
libosmvendor_RUN_REQ=""
libosmvendor_devel_RUN_REQ=""
perftest_RUN_REQ=""
mstflint_RUN_REQ=""
librdmacm_RUN_REQ=""
open_iscsi_RUN_REQ=""
mvapich_RUN_REQ=""
mvapich2_RUN_REQ=""
openmpi_RUN_REQ=""
mpitests_RUN_REQ=""
ofed_docs_RUN_REQ=""
libcxgb3_RUN_REQ=""
libcxgb3_devel_RUN_REQ=""
qlvnictools_RUN_REQ=""
sdpnetstat_RUN_REQ=""
ib_bonding_RUN_REQ=""
rds_tools_RUN_REQ=""
dapl_RUN_REQ=""
dapl_devel_RUN_REQ=""
dapl_utils_RUN_REQ=""

kernel_ib_BUILD_REQ=""
kernel_ib_devel_BUILD_REQ=""
libibverbs_BUILD_REQ="glibc-devel"
libibverbs_devel_BUILD_REQ=""
libibverbs_utils_BUILD_REQ=""
libibcm_BUILD_REQ=""
libmthca_BUILD_REQ=""
libmthca_devel_BUILD_REQ=""
libehca_BUILD_REQ=""
libehca_devel_BUILD_REQ=""
libipathverbs_BUILD_REQ=""
libipathverbs_devel_BUILD_REQ=""
libsdp_BUILD_REQ=""
srptools_BUILD_REQ=""
ipoibtools_BUILD_REQ=""
tvflash_BUILD_REQ="pciutils-devel"
libibcommon_BUILD_REQ=""
libibmad_BUILD_REQ=""
libibumad_BUILD_REQ=""
openib_diags_BUILD_REQ=""
ibutils_BUILD_REQ="tcl_8.4 tcl-devel_8.4 tk"
opensm_BUILD_REQ=""
libopensm_BUILD_REQ=""
libopensm_devel_BUILD_REQ=""
libosmcomp_BUILD_REQ=""
libosmcomp_devel_BUILD_REQ=""
libosmvendor_BUILD_REQ=""
libosmvendor_devel_BUILD_REQ=""
perftest_BUILD_REQ=""
mstflint_BUILD_REQ="zlib-devel"
librdmacm_BUILD_REQ=""
open_iscsi_BUILD_REQ=""
mvapich_BUILD_REQ=""    
mvapich2_BUILD_REQ=""
openmpi_BUILD_REQ=""    
mpitests_BUILD_REQ=""
ofed_docs_BUILD_REQ=""
libcxgb3_BUILD_REQ=""
libcxgb3_devel_BUILD_REQ=""
qlvnictools_BUILD_REQ=""
sdpnetstat_BUILD_REQ=""
ib_bonding_BUILD_REQ=""
rds_tools_BUILD_REQ=""
dapl_BUILD_REQ=""
dapl_devel_BUILD_REQ=""
dapl_utils_BUILD_REQ=""

case ${DISTRIBUTION} in
        SuSE)
        libibverbs_BUILD_REQ="${libibverbs_BUILD_REQ} rpm"
        case ${K_VER} in
                2.6.5*)
                mvapich2_BUILD_REQ="${mvapich2_BUILD_REQ} udev"
                openmpi_BUILD_REQ="${mvapich2_BUILD_REQ} udev"
                ;;
                *)
                mvapich2_BUILD_REQ="${mvapich2_BUILD_REQ} sysfsutils"
                openmpi_BUILD_REQ="${mvapich2_BUILD_REQ} sysfsutils"
                ;;
        esac
        ;;
        fedora6)
        # start_udev is required to create /dev/infiniband
        kernel_ib_RUN_REQ="${kernel_ib_RUN_REQ} udev"
        libibverbs_BUILD_REQ="${libibverbs_BUILD_REQ} rpm"
        mvapich2_BUILD_REQ="${mvapich2_BUILD_REQ} libsysfs-devel"
        openmpi_BUILD_REQ="${mvapich2_BUILD_REQ} libsysfs-devel"
        ;;
        debian)
        libibverbs_BUILD_REQ=
        libibverbs_devel_BUILD_REQ=
        ;;
        redhat5)
        # start_udev is required to create /dev/infiniband
        kernel_ib_RUN_REQ="${kernel_ib_RUN_REQ} udev"
        libibverbs_BUILD_REQ="${libibverbs_BUILD_REQ} rpm"
        mvapich2_BUILD_REQ="${mvapich2_BUILD_REQ} libsysfs-devel"
        openmpi_BUILD_REQ="${mvapich2_BUILD_REQ} libsysfs-devel"
        ;;
        *)
        libibverbs_BUILD_REQ="${libibverbs_BUILD_REQ} rpm-build"
        mvapich2_BUILD_REQ="${mvapich2_BUILD_REQ} sysfsutils-devel"
        openmpi_BUILD_REQ="${mvapich2_BUILD_REQ} sysfsutils-devel"
        ;;
esac


# Check the $1 package RPM Build Requirements ( Like kernel sources ... )
check_build_req()
{
        local RC=0
        local mode=$2
        local breq_pack
        local breq_pack_name
        local breq_pack_ver
        local breq_pack_rel
        local curr_pack_ver
            
        # Create cache files with RPM names and the list of capabilities
        for breq_pack in `eval echo \$\{${1}_BUILD_REQ\}`
        do
                if ( echo $breq_pack | grep '_' > $NULL 2>&1 ); then
                        breq_pack_name=`echo $breq_pack | cut -d '_' -f 1`
                        breq_pack_ver=`echo $breq_pack | cut -d '_' -f 2`
                        breq_pack_rel=`echo $breq_pack | cut -d '_' -f 3`
                else
                        breq_pack_name=$breq_pack
                        breq_pack_ver=""
                        breq_pack_rel=""
                fi    
                
                if ! ( grep -w $breq_pack_name ${rpms_cache} > $NULL 2>&1 ); then
                        [ "$mode" == "warn" ] && warn_echo "The $breq_pack_name package is required to build $1 RPM"
                        [ "$mode" == "err" ] && err_echo "The $breq_pack_name package is required to build $1 RPM"
                        let RC++
                else
                    if [ "$breq_pack_ver" != "" ]; then
                            curr_pack_ver=$($RPM --queryformat "[%{VERSION}]" -q $breq_pack_name)
                            compare $breq_pack_ver $curr_pack_ver
                            if [ $? -eq 1 ]; then
                                    [ "$mode" == "warn" ] && warn_echo "The $breq_pack_name >= $breq_pack_ver package is required to build $1"
                                    [ "$mode" == "err" ] && err_echo "The $breq_pack_name >= $breq_pack_ver package is required to build $1"
                                    let RC++
                            fi
                    fi
                
                fi
        done

        # Check for 64-bit libraries on 64-bit machines
        if [ "$ARCH" == "x86_64" ] || [ "$ARCH" == "ppc64" ]; then
            build_32bit=${build_32bit:-1}
            case $1 in
                libibverbs)
                    if [ ${build_32bit} -ne 0 ]; then
                        if ! ( $RPM -ql glibc-devel 2>&1 | grep "lib/crt1.o" > $NULL 2>&1 ) && 
                           ! ( $RPM -ql glibc-devel-32bit 2>&1 | grep "lib/crt1.o" > $NULL 2>&1 ); then
                                warn_echo "glibc-devel 32bit is required to build 32-bit libraries. Building 64-bit only..."
                                build_32bit=0
                        fi
                    fi
                ;;
                openmpi|mvapich|mvapich2|ibutils)
                    if ! ( grep libstdc++.so ${rpms_cache} | grep 64bit > $NULL 2>&1 ); then
                        err_echo "libstdc++ 64-bit version is required to build $1 packages"
                        let RC++
                    fi
                ;;
                mstflint)
                    if [ "$ARCH" == "ppc64" ]; then
                        if ! ( grep libstdc++.so ${rpms_cache} | grep -v 64bit > $NULL 2>&1 ); then
                            err_echo "libstdc++ 32-bit version is required to build $1 packages"
                            let RC++
                        fi
                    fi
                ;;
            esac
        else
            build_32bit=0
        fi

        return $RC    
}


# Select packages following their dependencies inside ${PACKAGE}
set_package_deps()
{        
        OFA_PACKAGES=""
        OFA_KERNEL_PACKAGES=""
        # IB_EXTRA_PACKAGES=""
        EXTRA_PACKAGES=""

        # Set default for low level driver
        # Can be defined as an environment variable or placed in the ofed.conf file for unattended installation
        ll_driver=${ll_driver:-"ib_mthca"}
        
        # Create list of packages to build
        for package in $SELECTED_PACKAGES
        do
                package_var=$(echo -n ${package} | sed -e 's/-/_/g')
                case ${package_var} in
                    ib_verbs)
                        OFA_KERNEL_PACKAGES=$(echo "$OFA_KERNEL_PACKAGES ib_verbs" | tr -s ' ' '\n' | sort -n | uniq)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    ib_mthca)
                        OFA_KERNEL_PACKAGES=$(echo "$OFA_KERNEL_PACKAGES ib_verbs ib_mthca" | tr -s ' ' '\n' | sort -n | uniq)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                        ll_driver=${ll_driver:-"ib_mthca"}
                    ;;
                    ib_ipath)
                        case ${ARCH} in
                                x86_64)
                                        case ${K_VER} in
                                                2.6.5*|2.6.9-22*|2.6.9-34*|2.6.9-42*|2.6.9-55*|2.6.16.*-*-*|2.6.1[7-9]*|2.6.20*)
                                                OFA_KERNEL_PACKAGES=$(echo "$OFA_KERNEL_PACKAGES ib_verbs ib_ipath" | tr -s ' ' '\n' | sort -n | uniq)
                                                OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                                                ll_driver=${ll_driver:-"ib_ipath"}
                                                ;;
                                                *)
                                                    if [ "$prog" == "build.sh" ]; then
                                                        warn_echo IPATH is not supported by this kernel
                                                    fi
                                                ;;
                                        esac
                                ;;
                                ppc64)
                                        case ${K_VER} in
                                                2.6.16.*-*-*|2.6.1[7-9]*|2.6.20*)
                                                OFA_KERNEL_PACKAGES=$(echo "$OFA_KERNEL_PACKAGES ib_verbs ib_ipath" | tr -s ' ' '\n' | sort -n | uniq)
                                                OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                                                ll_driver=${ll_driver:-"ib_ipath"}
                                                ;;
                                                *)
                                                    if [ "$prog" == "build.sh" ]; then
                                                        warn_echo IPATH is not supported by this kernel
                                                    fi
                                                ;;
                                        esac
                                ;;
                                *)
                                    if [ "$prog" == "build.sh" ]; then
                                        warn_echo PathScale InfiniPath host channel adapters low level driver supports only a x86_64 and ppc64 architectures
                                    fi
                                ;;
                        esac
                    ;;
                    ib_ehca)
                        case ${ARCH} in
                            ppc|ppc64|powerpc)
                                case ${K_VER} in
                                    2.6.1[6-9]* | 2.6.20* | 2.6.9-55*)
                                        OFA_KERNEL_PACKAGES=$(echo "$OFA_KERNEL_PACKAGES ib_verbs ib_ehca" | tr -s ' ' '\n' | sort -n | uniq)
                                        OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                                        ll_driver=${ll_driver:-"ib_ehca"}
                                    ;;
                                    *)
                                        if [ "$prog" == "build.sh" ]; then
                                            warn_echo EHCA is not supported by this kernel
                                        fi
                                    ;;
                                esac
                            ;;
                            *)
                                if [ "$prog" == "build.sh" ]; then
                                    warn_echo EHCA is not supported on ${ARCH}
                                fi
                            ;;
                        esac
                   ;;
                    ib_ipoib)
                            OFA_KERNEL_PACKAGES=$(echo "$OFA_KERNEL_PACKAGES ib_verbs ${ll_driver} ib_ipoib" | tr -s ' ' '\n' | sort -n | uniq)
                            OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    cxgb3)
                        case ${K_VER} in
                            2.6.13-1[5-9]*)
                                if [ "$prog" == "build.sh" ]; then
                                    warn_echo cxgb3 is not supported by this kernel
                                fi
                            ;;
                            *)
                                OFA_KERNEL_PACKAGES=$(echo "$OFA_KERNEL_PACKAGES ib_verbs ${ll_driver} cxgb3" | tr -s ' ' '\n' | sort -n | uniq)
                                OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                            ;;
                        esac
                    ;;
                    vnic)
                        case ${K_VER} in
                        2.6.19*|2.6.9-34*|2.6.9-42*|2.6.9-55*|2.6.16.*-*-*)
                            OFA_KERNEL_PACKAGES=$(echo "$OFA_KERNEL_PACKAGES ib_verbs ${ll_driver} vnic" | tr -s ' ' '\n' | sort -n | uniq)
                            OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                        ;;
                        *)
                            if [ "$prog" == "build.sh" ]; then
                                warn_echo VNIC is not supported by this kernel
                            fi
                        ;;
                        esac
                    ;;
                    ib_sdp)
                            OFA_KERNEL_PACKAGES=$(echo "$OFA_KERNEL_PACKAGES ib_verbs ${ll_driver} ib_ipoib ib_sdp" | tr -s ' ' '\n' | sort -n | uniq)
                            OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    ib_srp)
                            OFA_KERNEL_PACKAGES=$(echo "$OFA_KERNEL_PACKAGES ib_verbs ${ll_driver} ib_ipoib ib_srp" | tr -s ' ' '\n' | sort -n | uniq)
                            OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    ib_iser)
			# Currently iSER is supported only on SLES10 & RHEL5
                        case ${K_VER} in
                        2.6.16.*-*-*|2.6.*.el5|2.6.9-[3-5]*.EL*)
                            OFA_KERNEL_PACKAGES=$(echo "$OFA_KERNEL_PACKAGES ib_verbs ${ll_driver} ib_iser" | tr -s ' ' '\n' | sort -n | uniq)
                            OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                            EXTRA_PACKAGES=$(echo "$EXTRA_PACKAGES open-iscsi" | tr -s ' ' '\n' | sort -rn | uniq)
                        ;;
                        *)
                            if [ "$prog" == "build.sh" ]; then
                                warn_echo iSER is not supported by this kernel
                            fi
                        ;;
                        esac
                    ;;
                    rds)
                        case ${K_VER} in
                        2.6.5-7.*)
                            if [ "$prog" == "build.sh" ]; then
                                warn_echo RDS is not supported by this kernel
                            fi
                        ;;
                        *)
                            OFA_KERNEL_PACKAGES=$(echo "$OFA_KERNEL_PACKAGES ib_verbs ${ll_driver} ib_ipoib rds" | tr -s ' ' '\n' | sort -n | uniq)
                            OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                        ;;
                        esac
                    ;;
                    kernel_ib)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    kernel_ib_devel)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES kernel-ib-devel" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libibverbs)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libibverbs_devel)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libibverbs-devel" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libibverbs_utils)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libibverbs-utils" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libibcm)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libibcm" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libcxgb3)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libcxgb3" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libcxgb3_devel)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libcxgb3 libcxgb3-devel" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    qlvnictools)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libibumad libibcommon qlvnictools" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libibcm_devel)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libibcm libibcm-devel" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libmthca)
                        # Add libmthca package only if MTHCA driver selected to be installed
                        # if (echo -n ${SELECTED_PACKAGES} | grep -w "ib_mthca" > $NULL); then
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libmthca" | tr -s ' ' '\n' | sort -n | uniq)
                        # fi
                    ;;
                    libmthca_devel)
                        # Add libmthca-devel package only if MTHCA driver selected to be installed
                        # if (echo -n ${SELECTED_PACKAGES} | grep -w "ib_mthca" > $NULL); then
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libmthca libmthca-devel" | tr -s ' ' '\n' | sort -n | uniq)
                        # fi
                    ;;
                    libehca)
                        # Add libehca package only if EHCA driver selected to be installed
                        case ${ARCH} in
                            ppc|ppc64)
                                # if (echo -n ${SELECTED_PACKAGES} | grep -w "ib_ehca" > $NULL); then
                                OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libehca" | tr -s ' ' '\n' | sort -n | uniq)
                                # fi
                            ;;
                            *)
                                if [ "$prog" == "build.sh" ]; then
                                    warn_echo libehca is not supported on ${ARCH}
                                fi
                            ;;
                        esac
                    ;;
                    libehca_devel)
                        # Add libehca-devel package only if EHCA driver selected to be installed
                        case ${ARCH} in
                            ppc|ppc64)
                                # if (echo -n ${SELECTED_PACKAGES} | grep -w "ib_ehca" > $NULL); then
                                OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libehca libehca-devel" | tr -s ' ' '\n' | sort -n | uniq)
                                # fi
                            ;;
                            *)
                                if [ "$prog" == "build.sh" ]; then
                                    warn_echo libehca is not supported on ${ARCH}
                                fi
                            ;;
                        esac
                    ;;
                    libipathverbs)
                        # Add libipathverbs package only if IPATH driver selected to be installed
                        # if (echo -n ${SELECTED_PACKAGES} | grep -w "ib_ipath" > $NULL); then
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libipathverbs" | tr -s ' ' '\n' | sort -n | uniq)
                        # fi
                    ;;
                    libipathverbs_devel)
                        # Add libipathverbs-devel package only if IPATH driver selected to be installed
                        # if (echo -n ${SELECTED_PACKAGES} | grep -w "ib_ipath" > $NULL); then
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libipathverbs libipathverbs-devel" | tr -s ' ' '\n' | sort -n | uniq)
                        # fi
                    ;;
                    libsdp)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libsdp" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    srptools)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libibumad libibcommon srptools" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    ipoibtools)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES ipoibtools" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    sdpnetstat)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES sdpnetstat" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    ib_bonding)
                        # Add bonding support for RH4.0U3 and SLES10
                        case ${K_VER} in
                            2.6.16.*-*-*|2.6.9-34.EL*|2.6.9-42.EL*|2.6.9-55*|2.6.18-*.*)
                                EXTRA_PACKAGES=$(echo "$EXTRA_PACKAGES ib-bonding" | tr -s ' ' '\n' | sort -n | uniq)
                            ;;
                            *)
                                warn_echo "ib-bonding is not supported on this kernel"
                            ;;
                        esac
                    ;;
                    rds_tools)
                        EXTRA_PACKAGES=$(echo "$EXTRA_PACKAGES rds-tools" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    tvflash)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES tvflash" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libibcommon)
                        # libibverbs added as a workaround to the configure issue
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibverbs" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libibcommon_devel)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibcommon-devel libibverbs" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libibmad)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibmad libibumad" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libibmad_devel)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibmad libibmad-devel libibumad libibumad-devel" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libibumad)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibverbs libibumad" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libibumad_devel)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibverbs libibumad libibumad-devel" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    openib_diags)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibverbs libibmad libibumad libosmcomp libosmvendor libopensm openib-diags" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libosmcomp)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibverbs libibumad libosmcomp" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libosmcomp_devel)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibverbs libibumad libosmcomp-devel" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libosmvendor)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibverbs libibumad libosmvendor" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libosmvendor_devel)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibverbs libibumad libosmvendor-devel" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    opensm)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibverbs libibumad libosmcomp libosmvendor libopensm opensm" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libopensm)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibverbs libibumad libosmcomp libosmcomp-devel libosmvendor libosmvendor-devel libopensm" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    libopensm_devel)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibverbs libibumad libosmcomp libosmvendor libopensm-devel" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    ibutils)
                        EXTRA_PACKAGES=$(echo "$EXTRA_PACKAGES ibutils" | tr -s ' ' '\n' | sort -rn | uniq)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibcommon libibcommon-devel libibverbs libibverbs-devel libibmad libibmad-devel libibumad libibumad-devel libosmcomp libosmcomp-devel libosmvendor libosmvendor-devel libopensm libopensm-devel" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    perftest)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs librdmacm perftest" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    mstflint)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES mstflint" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    librdmacm)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs librdmacm" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    librdmacm_devel)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs librdmacm librdmacm-devel" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    librdmacm_utils)
                        OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs librdmacm librdmacm-utils" | tr -s ' ' '\n' | sort -n | uniq)
                    ;;
                    dapl)
                        case $(uname -m) in
                            ppc64)
                                if [ "$prog" == "build.sh" ]; then
                                    warn_echo "Dapl is not supported on the PPC64 arcitecture"
                                fi
                            ;;
                            *)
                            OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs librdmacm librdmacm-devel dapl" | tr -s ' ' '\n' | sort -n | uniq)
                            ;;
                        esac
                    ;;
                    dapl_devel)
                        case $(uname -m) in
                            ppc64)
                                if [ "$prog" == "build.sh" ]; then
                                    warn_echo "Dapl-devel is not supported on the PPC64 arcitecture"
                                fi
                            ;;
                            *)
                            OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs librdmacm librdmacm-devel dapl dapl-devel" | tr -s ' ' '\n' | sort -n | uniq)
                            ;;
                        esac
                    ;;
                    dapl_utils)
                        case $(uname -m) in
                            ppc64)
                                if [ "$prog" == "build.sh" ]; then
                                    warn_echo "Dapl-utils is not supported on the PPC64 arcitecture"
                                fi
                            ;;
                            *)
                            OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs librdmacm librdmacm-devel dapl dapl-devel dapl-utils" | tr -s ' ' '\n' | sort -n | uniq)
                            ;;
                        esac
                    ;;
                    open_iscsi)
                        # Add open-iscsi package only if iSER can be installed
                        check_open_iscsi
                        if (echo -n ${SELECTED_PACKAGES} | grep -w "ib_iser" > $NULL); then
                                EXTRA_PACKAGES=$(echo "$EXTRA_PACKAGES open-iscsi" | tr -s ' ' '\n' | sort -rn | uniq)
                        fi
                    ;;
                    mpi_selector)
                        EXTRA_PACKAGES=$(echo "$EXTRA_PACKAGES mpi-selector" | tr -s ' ' '\n' | sort -rn | uniq)
                    ;;
                    mvapich)
                            # Set compilation environment for OSU MPI                                    
                            if ! set_mpi_env mvapich; then
                                warn_echo "MVAPICH cannot be installed"
                                sed -i -e 's/mvapich=y/mvapich=n/g' $CONFIG
                                export mvapich=n
                            else                                 
                                EXTRA_PACKAGES=$(echo "$EXTRA_PACKAGES mpi-selector mvapich" | tr -s ' ' '\n' | sort -rn | uniq)
                                OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libibverbs-devel libibumad libibcommon" | tr -s ' ' '\n' | sort -n | uniq)
                            fi   
                    ;;
		    mvapich2)
                            # Set compilation environment for MVAPICH2
                            if ! set_mpi_env mvapich2; then
                                warn_echo "MVAPICH2 cannot be installed"
                                sed -i -e 's/mvapich2=y/mvapich2=n/g' $CONFIG
                                export mvapich2=n
                            else
                                EXTRA_PACKAGES=$(echo "$EXTRA_PACKAGES mpi-selector mvapich2" | tr -s ' ' '\n' | sort -rn | uniq)
                                if [ "$MVAPICH2_CONF_impl" = "ofa" ] && [ "$MVAPICH2_CONF_ckpt" = 0 ]; then
                                    # libibumad apparently needs libibcommon.
                                    OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libibverbs-devel libibumad libibumad-devel librdmacm librdmacm-devel libibcommon libibcommon-devel" | tr -s ' ' '\n' | sort -n | uniq)
                                elif [ "$MVAPICH2_CONF_impl" = "ofa" ]; then
                                    # Checkpoint-Restart does not support
                                    # RDMA-CM, so it would not be required.
                                    # libibumad apparently needs libibcommon.
                                    OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libibverbs-devel libibumad libibumad-devel libibcommon libibcommon-devel" | tr -s ' ' '\n' | sort -n | uniq)
                                elif [ "$MVAPICH2_CONF_impl" = "udapl" ]; then
                                    # dapl apparently needs libibverbs and
                                    # librdmacm.
                                    OFA_PACKAGES=$(echo "$OFA_PACKAGES dapl dapl-devel libibverbs librdmacm" | tr -s ' ' '\n' | sort -n | uniq)
                                fi
                            fi
                    ;;
                    openmpi)
                            # Set compilation environment for Open MPI
                            if ! set_mpi_env openmpi; then
                                warn_echo "Open MPI cannot be installed"
                                sed -i -e 's/openmpi=y/openmpi=n/g' $CONFIG
                                export openmpi=n
                            else                                     
                                EXTRA_PACKAGES=$(echo "$EXTRA_PACKAGES mpi-selector openmpi" | tr -s ' ' '\n' | sort -rn | uniq)
                                OFA_PACKAGES=$(echo "$OFA_PACKAGES libibverbs libibverbs-devel" | tr -s ' ' '\n' | sort -n | uniq)
                            fi

                    ;;
                    mpitests)
                        if (echo -n ${SELECTED_PACKAGES} | grep -Ew "mvapich|mvapich2|openmpi" > $NULL 2>&1); then
                            EXTRA_PACKAGES=$(echo "$EXTRA_PACKAGES mpitests" | tr -s ' ' '\n' | sort -rn | uniq)
                        fi
                    ;;
                    ofed_docs)
                        EXTRA_PACKAGES=$(echo "$EXTRA_PACKAGES ofed-docs" | tr -s ' ' '\n' | sort -rn | uniq)
                    ;;
                    ofed_scripts)
                        EXTRA_PACKAGES=$(echo "$EXTRA_PACKAGES ofed-scripts" | tr -s ' ' '\n' | sort -rn | uniq)
                    ;;
                    *) 
                        err_echo "Unknown package ${package}. Fix the ${CONFIG} file. Skipping ..."
                    ;;
                esac
        done

        EXTRA_PACKAGES=$(echo -n $EXTRA_PACKAGES | tr -s '\n' ' ')
	
        # Build the EXTRA_PACKAGES list from scratch following interpackage dependencies
        # in ALL_PACKAGES which has in the right order
        tmp_EXTRA_PACKAGES=""
        for pack in ${ALL_PACKAGES} ${HIDDEN_PACKAGES}
        do
            if ( echo -n ${EXTRA_PACKAGES} | grep -w ${pack} > $NULL 2>&1 ); then
                tmp_EXTRA_PACKAGES="${tmp_EXTRA_PACKAGES} ${pack}"
            fi
        done

        EXTRA_PACKAGES="${tmp_EXTRA_PACKAGES}"
	
        OFA_PACKAGES=$(echo -n $OFA_PACKAGES | tr -s '\n' ' ')
        OFA_KERNEL_PACKAGES=$(echo -n $OFA_KERNEL_PACKAGES | tr -s '\n' ' ')
        
        # SELECTED_PACKAGES="$OFA_PACKAGES $EXTRA_PACKAGES"
        if [ $INTERACTIVE -eq 1 ]; then
            echo
            pass_echo "Below is the list of ${PACKAGE} packages that you have chosen
(some may have been added by the installer due to package dependencies):"
            echo "$OFA_KERNEL_PACKAGES $OFA_PACKAGES $EXTRA_PACKAGES" | tr -s ' ' '\n'
        fi    

}



# Checking dependencies for selected packages
check_build_dependencies()
{
        echo
        echo "Checking dependencies. Please wait ..."
        echo

        # Check if kernel sources exists (needed for packages compilation)
        if [ ! -d ${K_SRC} ] || [ ! -f ${K_SRC}/Makefile ]; then
            echo
            err_echo "There are no sources for the ${K_VER} kernel installed.
                       Please install ${K_VER} kernel sources to build RPMs on this system"
            echo
            exit 1
        fi
        
        
        # Check dependencies and print Warning messages
        local req_status=0

        touch ${rpms_cache}

        $RPM -qa > ${rpms_cache} 2>&1
        $RPM -qa --provides >> ${rpms_cache} 2>&1

        for package in $OFA_PACKAGES $EXTRA_PACKAGES
        do
            package_var=$(echo -n ${package} | sed -e 's/-/_/g')
            check_build_req $package_var err
            let req_status=$req_status+$?
        done

        # Exit if there is missing required software for mandatory packages
        if [ $req_status -gt 0 ]; then
#           getch
           exit 7
        fi   

}

check_run_dependencies()
{
        # Check dependencies and print Warning messages
        local req_status=0
        local iscsi_err=0
        
        touch ${rpms_cache}

        $RPM -qa > ${rpms_cache} 2>&1
        $RPM -qa --provides >> ${rpms_cache} 2>&1

        for package in $OFA_PACKAGES $EXTRA_PACKAGES
        do
                package_var=$(echo -n ${package} | sed -e 's/-/_/g')
                check_run_req $package_var err
                let req_status=$req_status+$?
                if [ "${package_var}" == "open_iscsi" ]; then
                        check_open_iscsi
                fi
        done
        
        # Exit if there is missing required software for mandatory packages
        if [ $req_status -gt 0 ]; then
           exit 7
        fi   
}
 
include_ifcfg()
{
    if ( echo "$OFA_KERNEL_PACKAGES" | grep ipoib > $NULL 2>&1 ); then
        read -p "Do you want to include IPoIB configuration files (ifcfg-ib*)? [Y/n]:" ans_ip
        if ! [[ "$ans_ip" == "" || "$ans_ip" == "y" || "$ans_ip" == "Y" || "$ans_ip" == "yes" ]]; then
            export OFA_KERNEL_PARAMS="$OFA_KERNEL_PARAMS --without-ipoibconf"
            export INCLUDE_IFCFG=0
        fi
    fi
}
